import 'dart:convert';
import 'dart:io';
import 'package:flutter/material.dart';
import 'package:get/get.dart';
import 'package:path_provider/path_provider.dart';
import 'package:permission_handler/permission_handler.dart';
import 'package:viserpay_agent/core/utils/my_strings.dart';
import 'package:viserpay_agent/core/utils/url_container.dart';
import 'package:viserpay_agent/data/model/global/response_model/response_model.dart';
import 'package:viserpay_agent/data/model/qr_code/qr_code_download_response_model.dart';
import 'package:viserpay_agent/data/model/qr_code/qr_code_response_model.dart';
import 'package:viserpay_agent/data/model/qr_code/qr_code_scan_response_model.dart';
import 'package:viserpay_agent/data/repo/qr_code/qr_code_repo.dart';
import 'package:viserpay_agent/view/components/file_download_dialog/download_dialogue.dart';
import 'package:viserpay_agent/view/components/snack_bar/show_custom_snackbar.dart';
import 'package:share_plus/share_plus.dart';
import 'package:http/http.dart' as http;

class QrCodeController extends GetxController {
  QrCodeRepo qrCodeRepo;
  QrCodeController({required this.qrCodeRepo});

  bool isLoading = true;
  QrCodeResponseModel model = QrCodeResponseModel();

  String qrCode = "";
  String username = '';

  Future<void> loadData() async {
    username = qrCodeRepo.apiClient.getCurrencyOrUsername(isCurrency: false);
    isLoading = true;
    update();

    ResponseModel responseModel = await qrCodeRepo.getQrData();
    if (responseModel.statusCode == 200) {
      model = QrCodeResponseModel.fromJson(jsonDecode(responseModel.responseJson));
      if (model.status.toString().toLowerCase() == "success") {
        qrCode = model.data?.qrCode ?? "";
      } else {
        CustomSnackBar.error(errorList: model.message?.error ?? [MyStrings.requestFail]);
      }
    } else {
      CustomSnackBar.error(errorList: [responseModel.message]);
    }

    isLoading = false;
    update();
  }

  Future<void> shareImage() async {
    final box = Get.context!.findRenderObject() as RenderBox?;

    await Share.share(
      qrCode,
      subject: MyStrings.share.tr,
      sharePositionOrigin: box!.localToGlobal(Offset.zero) & box.size,
    );
  }

  String downloadUrl = "";
  String downloadFileName = "";
  bool downloadLoading = false;
  Future<void> downloadImage() async {
    downloadLoading = true;
    update();
    final headers = {
      'Authorization': "Bearer ${qrCodeRepo.apiClient.token}",
      'content-type': 'image/png',
    };

    String url = "${UrlContainer.baseUrl}${UrlContainer.qrCodeImageDownload}";
    http.Response response = await http.post(Uri.parse(url), body: null, headers: headers);
    final bytes = response.bodyBytes;
    String extension = "png";

    await saveAndOpenFile(bytes, '${MyStrings.appName}_${DateTime.now().millisecondsSinceEpoch}.$extension', extension);

    return;
  }

  Future<void> saveAndOpenFile(List<int> bytes, String fileName, String extension) async {
    Directory? downloadsDirectory;

    if (Platform.isAndroid) {
      await Permission.storage.request();
      downloadsDirectory = Directory('/storage/emulated/0/Download');
    } else if (Platform.isIOS) {
      downloadsDirectory = await getApplicationDocumentsDirectory();
    }

    if (downloadsDirectory != null) {
      final downloadPath = '${downloadsDirectory.path}/$fileName';
      final file = File(downloadPath);
      await file.writeAsBytes(bytes);
      CustomSnackBar.success(successList: ['File saved at: $downloadPath']);
    } else {
      CustomSnackBar.error(errorList: ["error"]);
    }
    downloadLoading = false;
    update();
  }

  bool isScannerLoading = false;
  Future<bool> submitQrData({
    required String scannedData,
    String expectedType = "-1",
    String nextRouteName = "-1",
  }) async {
    isScannerLoading = true;
    update();

    bool requestStatus = false;

    ResponseModel responseModel = await qrCodeRepo.qrCodeScan(scannedData);
    if (responseModel.statusCode == 200) {
      QrCodeSubmitScanResponseModel scanModel = QrCodeSubmitScanResponseModel.fromJson(jsonDecode(responseModel.responseJson));
      if (scanModel.status.toString().toLowerCase() == "success") {
        String userName = scanModel.data?.userData?.username ?? "";
        String userType = scanModel.data?.userType ?? "";
        String userNumber = scanModel.data?.userData?.mobile ?? "";

        if (userType.toUpperCase() == "USER") {
          Get.back(result: userName);
        } else {
          Get.back(result: "-1");
        }
      } else {
        Get.back(result: "-1");
        requestStatus = false;
        CustomSnackBar.error(errorList: model.message?.error ?? [MyStrings.requestFail]);
      }
    } else {
      requestStatus = false;
      CustomSnackBar.error(errorList: [responseModel.message]);
    }

    isScannerLoading = false;
    update();

    return requestStatus;
  }
  //
}
